/*
 * Copyright 2013-2014 Aaron Sowry for Cendio AB
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "includes.h"

#if defined(GSSAPI) && defined(USE_GSS_WRAP)
#include <stdio.h>
#include <stdbool.h>
#include <dlfcn.h>

#include "ssh-gss.h"

void *gss_handle = NULL;

static gss_OID_desc _GSS_C_NT_HOSTBASED_SERVICE = {10, (void *)"\x2a\x86\x48\x86\xf7\x12\x01\x02\x01\x04"};
gss_OID GSS_C_NT_HOSTBASED_SERVICE = &_GSS_C_NT_HOSTBASED_SERVICE;

/* Function pointer definitions
 * Taken from RFC 2744 (http://tools.ietf.org/html/rfc2744.html) */
OM_uint32 (*_gss_release_buffer)    (OM_uint32*,         gss_buffer_t);
OM_uint32 (*_gss_release_cred)      (OM_uint32*,         gss_cred_id_t*);
OM_uint32 (*_gss_release_name)      (OM_uint32*,         gss_name_t*);
OM_uint32 (*_gss_release_oid_set)   (OM_uint32*,         gss_OID_set *set);
OM_uint32 (*_gss_import_name)       (OM_uint32*,         const gss_buffer_t,
                                     const gss_OID,      gss_name_t*);
OM_uint32 (*_gss_display_status)    (OM_uint32*,         OM_uint32,
                                     int,                const gss_OID,
                                     OM_uint32*,         gss_buffer_t);
OM_uint32 (*_gss_indicate_mechs)    (OM_uint32*,         gss_OID_set*);
OM_uint32 (*_gss_get_mic)           (OM_uint32*,         const gss_ctx_id_t,
                                     gss_qop_t,          const gss_buffer_t,
                                     gss_buffer_t);
OM_uint32 (*_gss_delete_sec_context)(OM_uint32*,         gss_ctx_id_t*,
                                     gss_buffer_t);
OM_uint32 (*_gss_init_sec_context)  (OM_uint32*,         gss_cred_id_t,
                                     gss_ctx_id_t*,      gss_name_t,
                                     gss_OID,            OM_uint32,
                                     OM_uint32,          gss_channel_bindings_t,
                                     gss_buffer_t,       gss_OID*,
                                     gss_buffer_t,       OM_uint32*,
                                     OM_uint32*);

typedef struct fn_ptr_map_def {
	void **fn_ptr;
	const char *fn_name;
} fn_ptr_map;

/* Create a mapping of function names to the actual library functions */
const fn_ptr_map gss_fn_map[] = {
	{(void**)&_gss_release_buffer,     "gss_release_buffer"},
	{(void**)&_gss_release_cred,       "gss_release_cred"},
	{(void**)&_gss_release_name,       "gss_release_name"},
	{(void**)&_gss_release_oid_set,    "gss_release_oid_set"},
	{(void**)&_gss_import_name,        "gss_import_name"},
	{(void**)&_gss_display_status,     "gss_display_status"},
	{(void**)&_gss_indicate_mechs,     "gss_indicate_mechs"},
	{(void**)&_gss_get_mic,            "gss_get_mic"},
	{(void**)&_gss_delete_sec_context, "gss_delete_sec_context"},
	{(void**)&_gss_init_sec_context,   "gss_init_sec_context"}
};

static bool do_dlopen() {
	char *gss_lib = "libgssapi_krb5.so.2";    

	if (gss_handle != NULL)
		return true;

	/* Open GSSAPI library */
	gss_handle = dlopen(gss_lib, RTLD_NOW);

	if (gss_handle == NULL) {
		fprintf(stderr, "Error loading GSSAPI library: %s\n", dlerror());
		return false;
	}

	/* Assign function pointers */
	int n = sizeof(gss_fn_map) / sizeof(gss_fn_map[0]);

	int i;
	for (i = 0; i < n; i++) {
		*gss_fn_map[i].fn_ptr = dlsym(gss_handle, gss_fn_map[i].fn_name);
		if (! *gss_fn_map[i].fn_ptr) {
			fprintf(stderr, "Error loading GSSAPI library: %s\n", dlerror());
			return false;
		}
	}

	return true;
}

OM_uint32 gss_release_buffer (
        OM_uint32    *minor_status,
        gss_buffer_t buffer)
{
	if (!do_dlopen())
		return GSS_S_FAILURE;

	return _gss_release_buffer(minor_status, buffer);
};

OM_uint32 gss_release_cred (
        OM_uint32     *minor_status,
        gss_cred_id_t *cred_handle)
{
	if (!do_dlopen())
		return GSS_S_FAILURE;

	return _gss_release_cred(minor_status, cred_handle);
};

OM_uint32 gss_release_name (
        OM_uint32  *minor_status,
        gss_name_t *name)
{
	if (!do_dlopen())
		return GSS_S_FAILURE;

	return _gss_release_name(minor_status, name);
};

OM_uint32 gss_release_oid_set (
        OM_uint32   *minor_status,
        gss_OID_set *set)
{
	if (!do_dlopen())
		return GSS_S_FAILURE;

	return _gss_release_oid_set(minor_status, set);
}

OM_uint32 gss_import_name (
        OM_uint32          *minor_status,
        const gss_buffer_t input_name_buffer,
        const gss_OID      input_name_type,
        gss_name_t         *output_name)
{
	if (!do_dlopen())
		return GSS_S_FAILURE;

	return _gss_import_name(minor_status,    input_name_buffer,
	                        input_name_type, output_name);
};

OM_uint32 gss_display_status (
        OM_uint32      *minor_status,
        OM_uint32      status_value,
        int            status_type,
        const gss_OID  mech_type,
        OM_uint32      *message_context,
        gss_buffer_t   status_string)
{
	if(!do_dlopen())
		return GSS_S_FAILURE;

	return _gss_display_status(minor_status,    status_value,
	                           status_type,     mech_type,
	                           message_context, status_string);
};

/* OpenSSH expects gss_indicate_mechs to assign some valid pointer 
 * to mech_set, even if the call fails. Define an empty set which
 * we can pass back in this case. */
gss_OID_set_desc empty_set = {
	.count = 0,
	.elements = NULL
};

OM_uint32 gss_indicate_mechs (
        OM_uint32   *minor_status,
        gss_OID_set *mech_set)
{
	if (!do_dlopen()) {
		*mech_set = &empty_set;
		return GSS_S_FAILURE;
	}

	return _gss_indicate_mechs(minor_status, mech_set);
};

OM_uint32 gss_get_mic (
        OM_uint32          *minor_status,
        const gss_ctx_id_t context_handle,
        gss_qop_t          qop_req,
        const gss_buffer_t message_buffer,
        gss_buffer_t       msg_token)
{
	if (!do_dlopen())
		return GSS_S_FAILURE;

	return _gss_get_mic(minor_status, context_handle,
	                    qop_req,      message_buffer,
	                    msg_token);
};

OM_uint32 gss_delete_sec_context (
        OM_uint32    *minor_status,
        gss_ctx_id_t *context_handle,
        gss_buffer_t output_token)
{
	if (!do_dlopen())
		return GSS_S_FAILURE;

	return _gss_delete_sec_context(minor_status, context_handle,
	                               output_token);
};

OM_uint32 gss_init_sec_context (
        OM_uint32              *minor_status,
        gss_cred_id_t          initiator_cred_handle,
        gss_ctx_id_t           *context_handle,
        gss_name_t             target_name,
        gss_OID                mech_type,
        OM_uint32              req_flags,
        OM_uint32              time_req,
        gss_channel_bindings_t input_chan_bindings,
        gss_buffer_t           input_token,
        gss_OID                *actual_mech_type,
        gss_buffer_t           output_token,
        OM_uint32              *ret_flags,
        OM_uint32              *time_rec)
{
	if (!do_dlopen())
		return GSS_S_FAILURE;

	return _gss_init_sec_context(minor_status,   initiator_cred_handle,
	                             context_handle, target_name,
	                             mech_type,      req_flags,
	                             time_req,       input_chan_bindings,
	                             input_token,    actual_mech_type,
	                             output_token,   ret_flags,
	                             time_rec);
};

#endif /* GSSAPI && USE_GSS_WRAP */
