/*
 * Copyright 2013-2014 Pierre Ossman for Cendio AB
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "includes.h"

#undef strerror

#ifdef WIN32

#include <winternl.h>
#include <ntstatus.h>

int
isafifo(int fd)
{
	int ret;
	struct stat st;

	ret = fstat(fd, &st);
	if (ret != 0)
		return 0;

	return S_ISFIFO(st.st_mode);
}

int
fcntl(int fd, int cmd, ... /* arg */ )
{
	unsigned long val;
	int newfd;

	va_list ap;

	switch (cmd) {
	case F_SETFD:
		return 0;
	case F_GETFL:
		if (isatty(fd)) {
			/* The console cannot be made non-blocking */
			return 0;
		} else if (isafifo(fd)) {
			HANDLE handle;
			DWORD state;

			handle = (HANDLE)_get_osfhandle(fd);
			if (!GetNamedPipeHandleState(handle, &state, NULL,
			                             NULL, NULL, NULL, 0)) {
				errno = get_win32_error();
				return -1;
			}

			if (state & PIPE_NOWAIT)
				return O_NONBLOCK;

			return 0;
		} else {
			/*
			 * Sockets are blocking by default, but we have no
			 * way of checking what it is now.
			 */
			return 0;
		}
	case F_SETFL:
		va_start(ap, cmd);
		val = va_arg(ap, int);
		va_end(ap);

		if (isatty(fd)) {
			/* The console cannot be made non-blocking */
			if (val & O_NONBLOCK) {
				errno = EPERM;
				return -1;
			}

			return 0;
		} else if (isafifo(fd)) {
			int i;
			HANDLE handle;
			DWORD state;

			handle = (HANDLE)_get_osfhandle(fd);

			/*
			 * You can't reconfigure a pipe that has data in it
			 * so we need to check the buffers. We can't do
			 * anything about incoming data, but outgoing data
			 * we can wait a while for...
			 */
			for (i = 0;i < 100;i++) {
				IO_STATUS_BLOCK iosb;
				FILE_PIPE_LOCAL_INFORMATION fpli;
				NTSTATUS ret;

				/* From
				   https://msdn.microsoft.com/en-us/library/windows/hardware/ff567052%28v=vs.85%29.aspx:
				   Note If the call to this function
				   occurs in user mode, you should use
				   the name "NtQueryInformationFile"
				   instead of
				   "ZwQueryInformationFile". */
				ret = NtQueryInformationFile(handle, &iosb,
					                     &fpli, sizeof(fpli),
					                     FilePipeLocalInformation);
				if (ret != STATUS_SUCCESS) {
					/* We could use
					   LsaNtStatusToWinError, but
					   would need to link with
					   Advapi32. Use generic error. */
					SetLastError(ERROR_INVALID_PARAMETER);
					errno = get_win32_error();
					return -1;
				}

				if (fpli.WriteQuotaAvailable == fpli.OutboundQuota)
					break;

				Sleep(50);
			}

			if (!GetNamedPipeHandleState(handle, &state, NULL,
			                             NULL, NULL, NULL, 0)) {
				errno = get_win32_error();
				return -1;
			}

			if (val & O_NONBLOCK)
				state |= PIPE_NOWAIT;
			else
				state &= ~PIPE_NOWAIT;

			if (!SetNamedPipeHandleState(handle, &state,
			                             NULL, NULL)) {
				errno = get_win32_error();
				return -1;
			}

			return 0;
		} else {
			int ret;

			val = (val & O_NONBLOCK) != 0;

			ret = ioctlsocket(fd, FIONBIO, &val);
			if (ret == SOCKET_ERROR) {
				errno = get_win32_error();
				return -1;
			}

			return 0;
		}
	case F_DUPFD:
		va_start(ap, cmd);
		newfd = va_arg(ap, int);
		va_end(ap);

		while (1) {
			int ret;
			struct stat st;
			
			ret = fstat(newfd, &st);
			if ((ret != 0) && (errno == EBADF))
				break;

			newfd++;
		}

		return dup2(fd, newfd);
	default:
		errno = EINVAL;
		return -1;
	}
}

unsigned int
sleep(unsigned int seconds)
{
	Sleep(seconds * 1000);
	return 0;
}

char *
win32_strerror(int errnum)
{
	if (errnum > WSABASEERR) {
		static char msg[256];
		errnum &= ~WIN32_ERRNO;
		if (!FormatMessage(FORMAT_MESSAGE_FROM_SYSTEM, NULL, errnum,
		                   0, msg, sizeof(msg), NULL)) {
			snprintf(msg, sizeof(msg), "Unknown error %d", errnum);
		} else {
			// Windows likes line ending on their error messages
			if (strlen(msg) >= 2)
				msg[strlen(msg)-2] = '\0';
		}
		return msg;
	}	

	return strerror(errnum);
}

#endif /* WIN32 */
