/*
 * Copyright 2013-2014 Pierre Ossman for Cendio AB
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "includes.h"

#ifdef WIN32

#include <shlobj.h>

#define MAGIC_UID 666
#define MAGIC_GID 777

uid_t
getuid(void)
{
	return MAGIC_UID;
}

uid_t
geteuid(void)
{
	return getuid();
}

int
setuid(uid_t uid)
{
	if (uid != MAGIC_UID) {
		errno = EPERM;
		return -1;
	}

	return 0;
}

int
seteuid(uid_t euid)
{
	return setuid(euid);
}

struct passwd*
getpwnam(const char *name)
{
	struct passwd *pw;

	pw = getpwuid(MAGIC_UID);
	if (pw == NULL)
		return NULL;

	if (strcmp(pw->pw_name, name) != 0) {
		errno = ENOENT;
		return NULL;
	}

	return pw;
}

struct passwd*
getpwuid(uid_t uid)
{
	HRESULT result;

	static struct passwd pw;

	static char name[256];
	static char password[] = "";
	static char gecos[] = "";
	static char dir[MAX_PATH];
	static char shell[] = "/bin/false";

	DWORD len;

	if (uid != MAGIC_UID) {
		errno = ENOENT;
		return NULL;
	}

	memset(&pw, 0, sizeof(struct passwd));

	pw.pw_name = name;
	pw.pw_passwd = password;
	pw.pw_uid = MAGIC_UID;
	pw.pw_gid = MAGIC_GID;
	pw.pw_gecos = gecos;
	pw.pw_dir = dir;
	pw.pw_shell = shell;

	len = sizeof(name);
	if (!GetUserName(name, &len))
		return NULL;

	/*
	 * The Windows home directory is meant for user data, not
	 * configuration. So point things at the Application Data
	 * path instead.
	 */
	result = SHGetFolderPath(NULL, CSIDL_APPDATA, NULL, SHGFP_TYPE_CURRENT, dir);
	if (result != S_OK)
		return NULL;

	return &pw;
}

struct passwd*
getpwent(void)
{
	errno = ENOENT;
	return NULL;
}

void
setpwent(void)
{
}

void
endpwent(void)
{
}

struct group*
getgrnam(const char* name)
{
	errno = ENOENT;
	return NULL;
}

struct group*
getgrgid(gid_t gid)
{
	errno = ENOENT;
	return NULL;
}

struct group*
getgrent(void)
{
	errno = ENOENT;
	return NULL;
}

void
setgrent(void)
{
}

gid_t
getgid(void)
{
	return MAGIC_GID;
}

gid_t
getegid(void)
{
	return getgid();
}

int
setgid(gid_t gid)
{
	if (gid != MAGIC_GID) {
		errno = EPERM;
		return -1;
	}

	return 0;
}

int
setegid(gid_t egid)
{
	return setgid(egid);
}

int
initgroups(const char *user, gid_t group)
{
	errno = EPERM;
	return -1;
}

int
getgroups(int size, gid_t list[])
{
	return 0;
}

int
setgroups(size_t size, const gid_t *list)
{
	errno = EPERM;
	return -1;
}

#endif
