/***
  This file is part of PulseAudio.

  Copyright 2012 Pierre Ossman <ossman@cendio.se> for Cendio AB

  PulseAudio is free software; you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published
  by the Free Software Foundation; either version 2.1 of the License,
  or (at your option) any later version.

  PulseAudio is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with PulseAudio; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA.
***/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/stat.h>

#include <pulse/client-conf.h>
#include <pulsecore/core-util.h>

#include "module-simple-auto-symdef.h"

PA_MODULE_AUTHOR("Pierre Ossman");
PA_MODULE_DESCRIPTION("Try loading appropriate modules until one succeeds");
PA_MODULE_VERSION(PACKAGE_VERSION);
PA_MODULE_LOAD_ONCE(true);

static bool detect_pulse(void) {
    struct stat sb;
    char *ufn;
    pa_client_conf *conf;

    if (stat(PA_SYSTEM_RUNTIME_PATH PA_PATH_SEP PA_NATIVE_DEFAULT_UNIX_SOCKET, &sb) == 0)
        return true;

    if ((ufn = pa_runtime_path(PA_NATIVE_DEFAULT_UNIX_SOCKET))) {
        if (stat(ufn, &sb) == 0) {
            pa_xfree(ufn);
            return true;
        }
        pa_xfree(ufn);
    }

    conf = pa_client_conf_new();
    pa_client_conf_load(conf, true, true);

    if (conf->default_server) {
        pa_client_conf_free(conf);
        return true;
    }

    pa_client_conf_free(conf);

    return false;
}

int pa__init(pa_module*m) {
    pa_assert(m);

    if (detect_pulse()) {
        // Keep arguments in sync with tlclient
        if (!pa_module_load(m->core, "module-tunnel-sink-new",
                            "app_name='ThinLinc client' app_id='org.cendio.ThinLinc.client' app_version='' app_icon='thinlinc-client' sink='@DEFAULT_SINK@'"))
            return -1;
        if (!pa_module_load(m->core, "module-tunnel-source-new",
                            "app_name='ThinLinc client' app_id='org.cendio.ThinLinc.client' app_version='' app_icon='thinlinc-client' source='@DEFAULT_SOURCE@'"))
            return -1;
        goto done;
    }

#ifdef HAVE_ALSA
    if (pa_module_load(m->core, "module-alsa-sink", "")) {
        if (!pa_module_load(m->core, "module-alsa-source", ""))
            return -1;
        goto done;
    }
#endif

#ifdef HAVE_OSS_OUTPUT
    if (pa_module_load(m->core, "module-oss", ""))
        goto done;
#endif

    pa_log_warn("Failed to load any modules.");
    return -1;

done:
    /* We were successful and can unload ourselves now. */
    pa_module_unload_request(m, true);

    return 0;
}
