/***
  This file is part of PulseAudio.

  Copyright 2015 Pierre Ossman for Cendio AB

  PulseAudio is free software; you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published
  by the Free Software Foundation; either version 2.1 of the License,
  or (at your option) any later version.

  PulseAudio is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with PulseAudio; if not, see <http://www.gnu.org/licenses/>.
***/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <dlfcn.h>

#include <pulsecore/macro.h>

#include <xcb/xcb.h>
#include <X11/Xlib-xcb.h>

static void *libxcb;
static void *libX11_xcb;

typedef xcb_connection_t * (*xcb_connect_fn_t)(const char*, int*);
static xcb_connect_fn_t xcb_connect_fn;
typedef void (*xcb_disconnect_fn_t)(xcb_connection_t*);
static xcb_disconnect_fn_t xcb_disconnect_fn;
typedef int (*xcb_connection_has_error_fn_t)(xcb_connection_t*);
static xcb_connection_has_error_fn_t xcb_connection_has_error_fn;
typedef int (*xcb_flush_fn_t)(xcb_connection_t*);
static xcb_flush_fn_t xcb_flush_fn;
typedef xcb_intern_atom_cookie_t (*xcb_intern_atom_fn_t)(xcb_connection_t*,
                                                         uint8_t, uint16_t,
                                                         const char*);
static xcb_intern_atom_fn_t xcb_intern_atom_fn;
typedef xcb_intern_atom_reply_t* (*xcb_intern_atom_reply_fn_t)(xcb_connection_t*c,
                                                               xcb_intern_atom_cookie_t,
                                                               xcb_generic_error_t**);
static xcb_intern_atom_reply_fn_t xcb_intern_atom_reply_fn;
typedef const struct xcb_setup_t* (*xcb_get_setup_fn_t)(xcb_connection_t*);
static xcb_get_setup_fn_t xcb_get_setup_fn;
typedef xcb_screen_iterator_t (*xcb_setup_roots_iterator_fn_t)(const xcb_setup_t*);
static xcb_setup_roots_iterator_fn_t xcb_setup_roots_iterator_fn;
typedef void (*xcb_screen_next_fn_t)(xcb_screen_iterator_t*);
static xcb_screen_next_fn_t xcb_screen_next_fn;
typedef xcb_get_property_cookie_t (*xcb_get_property_fn_t)(xcb_connection_t*,
                                                           uint8_t,
                                                           xcb_window_t,
                                                           xcb_atom_t,
                                                           xcb_atom_t,
                                                           uint32_t,
                                                           uint32_t);
static xcb_get_property_fn_t xcb_get_property_fn;
typedef xcb_get_property_reply_t* (*xcb_get_property_reply_fn_t)(xcb_connection_t*,
                                                                 xcb_get_property_cookie_t,
                                                                 xcb_generic_error_t**);
static xcb_get_property_reply_fn_t xcb_get_property_reply_fn;
typedef void* (*xcb_get_property_value_fn_t)(const xcb_get_property_reply_t*);
static xcb_get_property_value_fn_t xcb_get_property_value_fn;
typedef int (*xcb_get_property_value_length_fn_t)(const xcb_get_property_reply_t*);
static xcb_get_property_value_length_fn_t xcb_get_property_value_length_fn;
typedef xcb_void_cookie_t (*xcb_change_property_fn_t)(xcb_connection_t*,
                                                      uint8_t,
                                                      xcb_window_t,
                                                      xcb_atom_t,
                                                      xcb_atom_t,
                                                      uint8_t,
                                                      uint32_t,
                                                      const void*);
static xcb_change_property_fn_t xcb_change_property_fn;
typedef xcb_void_cookie_t (*xcb_delete_property_fn_t)(xcb_connection_t*,
                                                      xcb_window_t,
                                                      xcb_atom_t);
static xcb_delete_property_fn_t xcb_delete_property_fn;

typedef xcb_connection_t* (*XGetXCBConnection_fn_t)(Display*);
static XGetXCBConnection_fn_t XGetXCBConnection_fn;

#define DLSYM_OR_FAIL(lib, name) \
    name ## _fn = (name ## _fn_t)dlsym(lib, #name); \
    if (!name ## _fn) { \
        pa_log_error("Could not find symbol \"%s\" in \"%s\": %s", \
                     #name, #lib, dlerror()); \
        goto fail; \
    }

static bool load_xcb(void) {
    if (libxcb)
        return true;

    libxcb = dlopen("libxcb.so.1", RTLD_LAZY);
    if (!libxcb) {
        pa_log_error("Could not load libxcb.so.1: %s", dlerror());
        return false;
    }

    DLSYM_OR_FAIL(libxcb, xcb_connect);
    DLSYM_OR_FAIL(libxcb, xcb_disconnect);
    DLSYM_OR_FAIL(libxcb, xcb_connection_has_error);
    DLSYM_OR_FAIL(libxcb, xcb_flush);
    DLSYM_OR_FAIL(libxcb, xcb_intern_atom);
    DLSYM_OR_FAIL(libxcb, xcb_intern_atom_reply);
    DLSYM_OR_FAIL(libxcb, xcb_get_setup);
    DLSYM_OR_FAIL(libxcb, xcb_setup_roots_iterator);
    DLSYM_OR_FAIL(libxcb, xcb_screen_next);
    DLSYM_OR_FAIL(libxcb, xcb_get_property);
    DLSYM_OR_FAIL(libxcb, xcb_get_property_reply);
    DLSYM_OR_FAIL(libxcb, xcb_get_property_value);
    DLSYM_OR_FAIL(libxcb, xcb_get_property_value_length);
    DLSYM_OR_FAIL(libxcb, xcb_change_property);
    DLSYM_OR_FAIL(libxcb, xcb_delete_property);

    return true;

fail:
    dlclose(libxcb);
    libxcb = NULL;

    return false;
}

static bool load_X11_xcb(void) {
    if (libX11_xcb)
        return true;

    libX11_xcb = dlopen("libX11-xcb.so.1", RTLD_LAZY);
    if (!libX11_xcb) {
        pa_log_error("Could not load libX11-xcb.so.1: %s", dlerror());
        return false;
    }

    DLSYM_OR_FAIL(libX11_xcb, XGetXCBConnection);

    return true;

fail:
    dlclose(libxcb);
    libxcb = NULL;

    return false;
}

xcb_connection_t *xcb_connect(const char *displayname, int *screenp) {
    if (!load_xcb())
        return NULL;
    return xcb_connect_fn(displayname, screenp);
}

void xcb_disconnect(xcb_connection_t *c) {
    if (!c)
        return;
    xcb_disconnect_fn(c);
}

int xcb_connection_has_error(xcb_connection_t *c) {
    pa_assert(c);
    return xcb_connection_has_error_fn(c);
}

int xcb_flush(xcb_connection_t *c) {
    pa_assert(c);
    return xcb_flush_fn(c);
}

const struct xcb_setup_t* xcb_get_setup(xcb_connection_t *c) {
    pa_assert(c);
    return xcb_get_setup_fn(c);
}

xcb_screen_iterator_t xcb_setup_roots_iterator(const xcb_setup_t *R) {
    pa_assert(R);
    return xcb_setup_roots_iterator_fn(R);
}

xcb_intern_atom_cookie_t xcb_intern_atom(xcb_connection_t *c,
                                         uint8_t only_if_exists,
                                         uint16_t name_len,
                                         const char *name) {
    pa_assert(c);
    return xcb_intern_atom_fn(c, only_if_exists, name_len, name);
}

xcb_intern_atom_reply_t *xcb_intern_atom_reply(xcb_connection_t *c,
                                               xcb_intern_atom_cookie_t cookie,
                                               xcb_generic_error_t **e) {
    pa_assert(c);
    return xcb_intern_atom_reply_fn(c, cookie, e);
}

void xcb_screen_next(xcb_screen_iterator_t *i) {
    pa_assert(i);
    xcb_screen_next_fn(i);
}

xcb_get_property_cookie_t xcb_get_property(xcb_connection_t *c,
uint8_t _delete,
xcb_window_t window,
xcb_atom_t property,
xcb_atom_t type,
uint32_t long_offset,
uint32_t long_length) {
    pa_assert(c);
    return xcb_get_property_fn(c, _delete, window, property, type,
                               long_offset, long_length);
}

xcb_get_property_reply_t *xcb_get_property_reply(xcb_connection_t *c,
xcb_get_property_cookie_t cookie,
xcb_generic_error_t **e) {
    pa_assert(c);
    return xcb_get_property_reply_fn(c, cookie, e);
}

void *xcb_get_property_value(const xcb_get_property_reply_t *R) {
    pa_assert(R);
    return xcb_get_property_value_fn(R);
}

int xcb_get_property_value_length(const xcb_get_property_reply_t *R) {
    pa_assert(R);
    return xcb_get_property_value_length_fn(R);
}

xcb_void_cookie_t xcb_change_property(xcb_connection_t *c,
                                      uint8_t mode,
                                      xcb_window_t window,
                                      xcb_atom_t property,
                                      xcb_atom_t type,
                                      uint8_t format,
                                      uint32_t data_len,
                                      const void *data) {
    pa_assert(c);
    return xcb_change_property_fn(c, mode, window, property, type,
                                  format, data_len, data);
}

xcb_void_cookie_t xcb_delete_property(xcb_connection_t *c,
                                      xcb_window_t window,
                                      xcb_atom_t property) {
    pa_assert(c);
    return xcb_delete_property_fn(c, window, property);
}

xcb_connection_t *XGetXCBConnection(Display *dpy) {
    if (!load_X11_xcb())
        return NULL;
    return XGetXCBConnection_fn(dpy);
}
