/*
 * Copyright 2013-2014 Pierre Ossman for Cendio AB
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "includes.h"

#include "log.h"

#ifdef WIN32

static FD_SET afunix_sockets;
static int skip_redirection;

int
win32_is_afunix(int fd)
{
	/* Temporary bypass */
	if (skip_redirection) {
		skip_redirection = 0;
		return 0;
	}

	return FD_ISSET(fd, &afunix_sockets);
}

int
win32_afunix_close(int fd)
{
	if (!win32_is_afunix(fd)) {
		errno = EINVAL;
		return -1;
	}

	FD_CLR(fd, &afunix_sockets);

	return 0;
}

int
win32_afunix_accept(int socket, struct sockaddr *address,
                    socklen_t *address_len)
{
	errno = EINVAL;
	return -1;
}

int
win32_afunix_bind(int socket, const struct sockaddr *address,
                  socklen_t address_len)
{
	errno = EINVAL;
	return -1;
}

int
win32_afunix_connect(int socket, const struct sockaddr *address,
                     socklen_t address_len)
{
	int ret;
	const struct sockaddr_un *addr;

	FILE *f;
	int port;
	char cookie[65];

	struct sockaddr_in sin;

	if (!win32_is_afunix(socket)) {
		errno = EINVAL;
		return -1;
	}

	if ((address == NULL) || (address_len != sizeof(struct sockaddr_un))) {
		errno = EINVAL;
		return -1;
	}

	addr = (const struct sockaddr_un*)address;

	f = fopen(addr->sun_path, "rt");
	if (f == NULL)
		return -1;

	port = 0;
	cookie[0] = '\0';

	fscanf(f, "%d ", &port);
	fgets(cookie, sizeof(cookie), f);
	
	fclose(f);

	if ((port == 0) || (strlen(cookie) != sizeof(cookie)-1)) {
		errno = EINVAL;
		return -1;
	}

	memset(&sin, 0, sizeof(struct sockaddr_in));

	sin.sin_family = AF_INET;
	sin.sin_port = htons(port);
	sin.sin_addr.s_addr = htonl(INADDR_LOOPBACK);

	/* Temporarily allow us to call a real connect() */
	skip_redirection = 1;

	ret = connect(socket, (struct sockaddr*)&sin,
	              sizeof(struct sockaddr_in));
	if (ret == -1) {
		FD_SET fds;

		if (errno != EINPROGRESS)
			return -1;

		debug("waiting for emulated unix socket to connect...");

		FD_ZERO(&fds);
		FD_SET(socket, &fds);

		ret = select(socket, NULL, &fds, NULL, NULL);
		if (ret == -1)
			return -1;

		debug("done.");
	}

	ret = write(socket, cookie, strlen(cookie));
	if (ret == -1)
		return -1;

	return 0;
}

int
win32_afunix_listen(int sockfd, int backlog)
{
	errno = EINVAL;
	return -1;
}

int
win32_afunix_socket(int domain, int type, int protocol)
{
	int fd;

	if ((domain != AF_UNIX) || (type != SOCK_STREAM)) {
		errno = EINVAL;
		return -1;
	}

	fd = socket(AF_INET, SOCK_STREAM, 0);
	if (fd == -1)
		return -1;

	FD_SET(fd, &afunix_sockets);

	return fd;
}

#endif /* WIN32 */
