/*
 * Copyright 2013-2024 Pierre Ossman for Cendio AB
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "includes.h"

#ifdef WIN32

typedef void (*sigfn)(int);

static sigset_t blocked_signals = 0;
static sigset_t pending_signals = 0;
static sigfn blocked_handlers[NSIG];

static void
queue_signal(int sig)
{
	sigaddset(&pending_signals, sig);
}

int
sigaction(int sig, struct sigaction *sigact, struct sigaction *osigact)
{
	int blocked;
	void (*old)(int);

	if (sig >= NSIG) {
		errno = EINVAL;
		return -1;
	}

	blocked = sigismember(&blocked_signals, sig);
	if (blocked == -1)
		return -1;

	if (blocked) {
		if (osigact)
			osigact->sa_handler = blocked_handlers[sig];

		if (sigact)
			blocked_handlers[sig] = sigact->sa_handler;

		return 0;
	}

	if (sigact)
		old = signal(sig, sigact->sa_handler);
	else
		old = signal(sig, SIG_IGN);

	if (old == SIG_ERR)
		return -1;

	if (!sigact)
		signal(sig, old);

	if (osigact)
		osigact->sa_handler = old;

	return 0;
}

int
sigemptyset(sigset_t *mask)
{
	if (mask == NULL) {
		errno = EINVAL;
		return -1;
	}

	*mask = 0;

	return 0;
}

int
sigfillset(sigset_t *mask)
{
	if (mask == NULL) {
		errno = EINVAL;
		return -1;
	}

	*mask = (sigset_t)-1;

	return 0;
}

int sigaddset(sigset_t *set, int signum)
{
	if (set == NULL) {
		errno = EINVAL;
		return -1;
	}

	if (signum >= sizeof(sigset_t)*8) {
		errno = EINVAL;
		return -1;
	}

	*set |= 1 << signum;

	return 0;
}

int sigdelset(sigset_t *set, int signum)
{
	if (set == NULL) {
		errno = EINVAL;
		return -1;
	}

	if (signum >= sizeof(sigset_t)*8) {
		errno = EINVAL;
		return -1;
	}

	*set &= ~(1 << signum);

	return 0;
}

int sigismember(const sigset_t *set, int signum)
{
	if (set == NULL) {
		errno = EINVAL;
		return -1;
	}

	if (signum >= sizeof(sigset_t)*8) {
		errno = EINVAL;
		return -1;
	}

	if (*set & (1 << signum))
		return 1;
	else
		return 0;
}

int sigprocmask(int how, const sigset_t *set, sigset_t *oldset)
{
	sigset_t newblock, unblock;

	if (set == NULL) {
		errno = EINVAL;
		return -1;
	}

	if (oldset != NULL)
		*oldset = blocked_signals;

	switch (how) {
	case SIG_BLOCK:
		newblock = *set & ~blocked_signals;
		unblock = 0;
		break;
	case SIG_UNBLOCK:
		newblock = 0;
		unblock = *set & blocked_signals;
		break;
	case SIG_SETMASK:
		newblock = *set & ~blocked_signals;
		unblock = ~*set & blocked_signals;
		break;
	default:
		errno = EINVAL;
		return -1;
	}

	for (int i = 0;i < NSIG;i++) {
		if (sigismember(&newblock, i)) {
			int ret;
			struct sigaction new, old;

			new.sa_handler = queue_signal;
			ret = sigaction(i, &new, &old);
			if (ret == -1)
				return -1;

			blocked_handlers[i] = old.sa_handler;
			sigaddset(&blocked_signals, i);
		}

		if (sigismember(&unblock, i)) {
			int ret;
			struct sigaction new;

			new.sa_handler = blocked_handlers[i];
			ret = sigaction(i, &new, NULL);
			if (ret == -1)
				return -1;

			blocked_handlers[i] = NULL;
			sigdelset(&blocked_signals, i);
		}
	}

	for (int i = 0;i < NSIG;i++) {
		if (sigismember(&unblock, i) &&
		    sigismember(&pending_signals, i)) {
			sigdelset(&pending_signals, i);
			raise(i);
		}
	}

	return 0;
}

int kill(pid_t pid, int sig)
{
	if (pid != getpid()) {
		errno = EPERM;
		return -1;
	}

	if (sig >= NSIG) {
		errno = EINVAL;
		return -1;
	}

	return raise(sig);
}

unsigned int
alarm(unsigned int seconds)
{
	return 0;
}

#endif /* WIN32 */