/*
 * Copyright 2013-2014 Aaron Sowry for Cendio AB
 *
 * Permission to use, copy, modify, and distribute this software for any
 * purpose with or without fee is hereby granted, provided that the above
 * copyright notice and this permission notice appear in all copies.
 *
 * THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
 * WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
 * ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
 * WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
 * ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
 * OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
 */

#include "includes.h"

#if defined(WIN32)
#include <string.h>
#define SECURITY_WIN32  1
#include <security.h>
#include <xmalloc.h>

#include "win32-sspi.h"

static gss_OID_desc _GSS_C_NT_HOSTBASED_SERVICE = {10, (void *)"\x2a\x86\x48\x86\xf7\x12\x01\x02\x01\x04"};
gss_OID GSS_C_NT_HOSTBASED_SERVICE = &_GSS_C_NT_HOSTBASED_SERVICE;

static gss_OID_desc _GSS_MECH_KRB5 = {9, (void *)"\x2A\x86\x48\x86\xF7\x12\x01\x02\x02"};
static gss_OID GSS_MECH_KRB5 = &_GSS_MECH_KRB5;

struct gss_ctx_id_struct {
	CtxtHandle *ctxt_handle;
	CredHandle *cred_handle;
};

OM_uint32 gss_display_status (
        OM_uint32      *minor_status,
        OM_uint32      status_value,
        int            status_type,
        const gss_OID  mech_type,
        OM_uint32      *message_context,
        gss_buffer_t   status_string)
{
	OM_uint32 tmp;
	char *msg;

	if (status_string == NULL || message_context == NULL)
		return GSS_S_FAILURE;

	status_string->length = 0;
	status_string->value = NULL;

	if (minor_status != NULL)
		*minor_status = 0;

	// XXX: Since this is Windows-specific, we are assuming
	//      a mech_type of GSS_MECH_KRB5.

	if (status_type == GSS_C_GSS_CODE) {
		// Major status
		if (status_value == 0) {
			*message_context = 0;
			return GSS_S_BAD_STATUS;
		}

		// Routine error
		if (*message_context == 0) {
			if ((tmp = GSS_ROUTINE_ERROR(status_value))) {
				status_value -= tmp;
				switch (tmp) {
					case GSS_S_FAILURE:
						msg = "Unspecified GSS failure. Minor code may provide more information";
						break;
					case GSS_S_BAD_MECH:
						msg = "An unsupported mechanism was requested";
						break;
					case GSS_S_BAD_NAME:
						msg = "An invalid name was supplied";
						break;
					case GSS_S_BAD_NAMETYPE:
						msg = "A supplied name was of an unsupported type";
						break;
					case GSS_S_BAD_BINDINGS:
						msg = "Incorrect channel bindings were supplied";
						break;
					case GSS_S_BAD_SIG:
						msg = "A token had an invalid Message Integrity Check (MIC)";
						break;
					case GSS_S_NO_CRED:
						msg = "No credentials were supplied, or the credentials were unavailable or inaccessible";
						break;
					case GSS_S_NO_CONTEXT:
						msg = "No context has been established";
						break;
					case GSS_S_DEFECTIVE_TOKEN:
						msg = "Invalid token was supplied";
						break;
					case GSS_S_DEFECTIVE_CREDENTIAL:
						msg = "Invalid credential was supplied";
						break;
					case GSS_S_CREDENTIALS_EXPIRED:
						msg = "The referenced credential has expired";
						break;
					case GSS_S_CONTEXT_EXPIRED:
						msg = "The referenced context has expired";
						break;
					case GSS_S_BAD_QOP:
						msg = "The quality-of-protection (QOP) requested could not be provided";
						break;
					case GSS_S_UNAUTHORIZED:
						msg = "The operation is forbidden by local security policy";
						break;
					case GSS_S_UNAVAILABLE:
						msg = "The operation or option is not available or unsupported";
						break;
					case GSS_S_DUPLICATE_ELEMENT:
						msg = "The requested credential element already exists";
						break;
					case GSS_S_NAME_NOT_MN:
						msg = "The provided name was not mechanism specific (MN)";
						break;
					case GSS_S_BAD_STATUS:
					default:
						msg = "An invalid status code was supplied";
				}
				status_string->length = strlen(msg);
				status_string->value = xstrdup(msg);
				if (status_value) {
					(*message_context)++;
					return GSS_S_COMPLETE;
				}
				else {
					*message_context = 0;
					return GSS_S_COMPLETE;
				}
			}
			else
				(*message_context)++;
		}
		else
			status_value -= GSS_ROUTINE_ERROR(status_value);
		
		// Calling error
		if (*message_context == 1) {
			if ((tmp = GSS_CALLING_ERROR(status_value))) {
				status_value -= tmp;
				switch (tmp) {
					case GSS_S_CALL_INACCESSIBLE_READ:
						msg = "A required input parameter could not be read";
						break;
					case GSS_S_CALL_INACCESSIBLE_WRITE:
						msg = "A required input parameter could not be written";
						break;
					case GSS_S_CALL_BAD_STRUCTURE:
						msg = "A parameter was malformed";
						break;
					default:
						msg = "An invalid status code was supplied";
				}
				status_string->length = strlen(msg);
				status_string->value = xstrdup(msg);
				if (status_value) {
					(*message_context)++;
					return GSS_S_COMPLETE;
				}
				else {
					*message_context = 0;
					return GSS_S_COMPLETE;
				}
			}
			else
				(*message_context)++;
		}
		else
			status_value -= GSS_CALLING_ERROR(status_value);

		// FIXME: Check supplimentary info bits as well

	}
	else if (status_type == GSS_C_MECH_CODE) {
		// Minor status
		if (*message_context)
			return GSS_S_FAILURE;

		switch (status_value) {
			// FIXME: We don't set a minor code on Windows
			default:
				msg = "No minor code available";
		}
		status_string->length = strlen(msg);
		status_string->value = xstrdup(msg);

		return GSS_S_COMPLETE;
	}
	else
		return GSS_S_BAD_STATUS;

	return GSS_S_COMPLETE;

};

OM_uint32 gss_release_buffer (
        OM_uint32    *minor_status,
        gss_buffer_t buffer)
{
	if (minor_status != NULL)
		*minor_status = 0;

	if (buffer == GSS_C_NO_BUFFER)
		return GSS_S_COMPLETE;

	if (buffer->value) {
		free(buffer->value);
		buffer->value = NULL;
		buffer->length = 0;
	}

	return GSS_S_COMPLETE;

};

OM_uint32 gss_delete_sec_context (
        OM_uint32    *minor_status,
        gss_ctx_id_t *context_handle,
        gss_buffer_t output_token)
{
	if (minor_status != NULL)
		*minor_status = 0;

	if (context_handle == NULL || *context_handle == GSS_C_NO_CONTEXT)
		return GSS_S_NO_CONTEXT;

	int ret;
	gss_ctx_id_t context;

	context = *context_handle;
	ret = DeleteSecurityContext(context->ctxt_handle);

	if (ret == SEC_E_OK) {
		if (context->cred_handle)
			free(context->cred_handle);
		free(context);
		*context_handle = GSS_C_NO_CONTEXT;
		return GSS_S_COMPLETE;
	}

	return GSS_S_FAILURE;

};

OM_uint32 gss_release_name (
        OM_uint32  *minor_status,
        gss_name_t *name)
{
	if (minor_status != NULL)
		*minor_status = 0;

	if (name == NULL)
		return GSS_S_BAD_NAME;

	if (*name == GSS_C_NO_NAME)
		return GSS_S_COMPLETE;

	free(*name);
	*name = GSS_C_NO_NAME;

	return GSS_S_COMPLETE;

};

OM_uint32 gss_release_cred (
        OM_uint32     *minor_status,
        gss_cred_id_t *cred_handle)
{
	if (minor_status != NULL)
		*minor_status = 0;

	if (cred_handle == NULL)
		return GSS_S_NO_CRED;

	if (*cred_handle == GSS_C_NO_CREDENTIAL)
		return GSS_S_COMPLETE;

	if (*cred_handle)
		free(*cred_handle);
	*cred_handle = GSS_C_NO_CREDENTIAL;

	return GSS_S_COMPLETE;

};

OM_uint32 gss_release_oid_set (
        OM_uint32   *minor_status,
        gss_OID_set *set)
{
	if (minor_status != NULL)
		*minor_status = 0;

	if (set == NULL)
		return GSS_S_COMPLETE;

	if (*set == GSS_C_NO_OID_SET)
		return GSS_S_COMPLETE;

	if (*set) {
		free((*set)->elements);
		free(*set);
	}
	*set = GSS_C_NO_OID_SET;

	return GSS_S_COMPLETE;
}

OM_uint32 gss_init_sec_context (
        OM_uint32              *minor_status,
        gss_cred_id_t          initiator_cred_handle,
        gss_ctx_id_t           *context_handle,
        gss_name_t             target_name,
        gss_OID                mech_type,
        OM_uint32              req_flags,
        OM_uint32              time_req,
        gss_channel_bindings_t input_chan_bindings,
        gss_buffer_t           input_token,
        gss_OID                *actual_mech_type,
        gss_buffer_t           output_token,
        OM_uint32              *ret_flags,
        OM_uint32              *time_rec)
{
	int ret;
	gss_ctx_id_t context;
	PCtxtHandle input_context;

	if (minor_status == NULL)
		return GSS_S_FAILURE | GSS_S_CALL_INACCESSIBLE_WRITE;

	*minor_status = 0;

	// Set up context handles
	context = *context_handle;

	if (context == GSS_C_NO_CONTEXT) {
		// This is the first call to gss_init_sec_context
		context = xcalloc(1, sizeof(struct gss_ctx_id_struct));
		context->cred_handle = xmalloc(sizeof(CredHandle));

		TimeStamp ptsExpiry;
		ret = AcquireCredentialsHandle(NULL,
		                               "Kerberos",
		                               SECPKG_CRED_OUTBOUND,
		                               NULL,
		                               NULL,
		                               NULL,
		                               NULL,
		                               context->cred_handle,
		                               &ptsExpiry);

		if (ret != SEC_E_OK) {
			free(context->cred_handle);
			free(context);

			if (ret == SEC_E_NO_CREDENTIALS ||
			    ret == SEC_E_UNKNOWN_CREDENTIALS)
				return GSS_S_NO_CRED;

			return GSS_S_FAILURE;
		}
	}

	// Set up input buffer
	SecBufferDesc input_desc;
	SecBuffer input_buffer;

	if (input_token == GSS_C_NO_BUFFER) {
		input_buffer.cbBuffer = 0;
		input_buffer.pvBuffer = NULL;
		input_buffer.BufferType = SECBUFFER_TOKEN;
	}
	else {
		input_buffer.cbBuffer = input_token->length;
		input_buffer.pvBuffer = input_token->value;
		input_buffer.BufferType = SECBUFFER_TOKEN;
	}

	input_desc.cBuffers = 1;
	input_desc.pBuffers = &input_buffer;
	input_desc.ulVersion = SECBUFFER_VERSION;

	// Set up output buffer
	SecBufferDesc output_desc;
	SecBuffer output_buffer;

	// Allocate our own token buffer
	PSecPkgInfo PackageInfo;
	QuerySecurityPackageInfo("Kerberos", &PackageInfo);

	output_buffer.cbBuffer = PackageInfo->cbMaxToken;
	output_buffer.pvBuffer = xmalloc(PackageInfo->cbMaxToken);
	output_buffer.BufferType = SECBUFFER_TOKEN;

	output_desc.cBuffers = 1;
	output_desc.pBuffers = &output_buffer;
	output_desc.ulVersion = SECBUFFER_VERSION;

	// Set up context attribute flags handle
	unsigned long output_flags;

	// Set up flags
	unsigned long flags = ISC_REQ_MUTUAL_AUTH|ISC_REQ_REPLAY_DETECT|
	                      ISC_REQ_CONFIDENTIALITY;

	if (req_flags & GSS_C_DELEG_FLAG)
		flags |= ISC_REQ_DELEGATE;

	// According to Microsoft documentation it's OK that the 9th argument
	// is the same as input_context in the second call:
	// > On the second call, phNewContext can be the same as the handle
	// > specified in the phContext parameter.
	if (context->ctxt_handle == NULL) {
		input_context = NULL;
		context->ctxt_handle = xmalloc(sizeof(CtxtHandle));
	} else {
		input_context = context->ctxt_handle;
	}

	ret = InitializeSecurityContext(context->cred_handle,
	                                input_context,
	                                (char *) target_name,
	                                flags,
	                                0, // Reserved
	                                SECURITY_NATIVE_DREP,
	                                &input_desc,
	                                0, // Reserved
	                                context->ctxt_handle,
	                                &output_desc,
	                                &output_flags,
	                                NULL);

	*context_handle = context;

	if (ret_flags != NULL)
		if (output_flags & ISC_RET_INTEGRITY)
			*ret_flags |= GSS_C_INTEG_FLAG;

	// FIXME: Check that the correct token type was received
	output_token->length = output_buffer.cbBuffer;
	output_token->value = output_buffer.pvBuffer;

	if (ret == SEC_I_CONTINUE_NEEDED)
		return GSS_S_CONTINUE_NEEDED;
	else if (ret == SEC_E_OK)
		return GSS_S_COMPLETE;

	return GSS_S_FAILURE;

};

OM_uint32 gss_import_name (
        OM_uint32          *minor_status,
        const gss_buffer_t input_name_buffer,
        const gss_OID      input_name_type,
        gss_name_t         *output_name)
{
	if (minor_status == NULL)
		return GSS_S_FAILURE | GSS_S_CALL_INACCESSIBLE_WRITE;

	*minor_status = 0;

	if (output_name == NULL)
		return GSS_S_FAILURE | GSS_S_CALL_INACCESSIBLE_WRITE;

	char *sspi_name;

	if (input_name_type == GSS_C_NT_HOSTBASED_SERVICE) {
		char buf[1024];
		char host[1024];
		char *ch;

		if (input_name_buffer->length + 1 > sizeof(buf))
			return GSS_S_BAD_NAME;

		memcpy(buf, input_name_buffer->value, input_name_buffer->length);
		buf[input_name_buffer->length] = '\0';
		ch = strchr(buf, '@');

		if (ch == NULL)
			return GSS_S_BAD_NAME;

		strcpy(host, ch+1);

		xasprintf(&sspi_name, "host/%s", host);
	}
	else
		// Unsupported name type
		return GSS_S_BAD_NAMETYPE;

	*output_name = (gss_name_t) sspi_name;

	return GSS_S_COMPLETE;

};

OM_uint32 gss_get_mic (
        OM_uint32          *minor_status,
        const gss_ctx_id_t context_handle,
        gss_qop_t          qop_req,
        const gss_buffer_t message_buffer,
        gss_buffer_t       msg_token)
{
	if (minor_status == NULL)
		return GSS_S_FAILURE | GSS_S_CALL_INACCESSIBLE_WRITE;
	   
	*minor_status = 0;

	if (msg_token == NULL)
		return GSS_S_FAILURE | GSS_S_CALL_INACCESSIBLE_WRITE;

	if (message_buffer == NULL)
		return GSS_S_FAILURE | GSS_S_CALL_INACCESSIBLE_READ;

	if (context_handle == GSS_C_NO_CONTEXT)
		return GSS_S_NO_CONTEXT;

	SecPkgContext_Sizes context_sizes;
	memset(&context_sizes, 0, sizeof(context_sizes));

	SecBufferDesc input_desc;
	SecBuffer input_security_token[2];

	int ret;
	ret = QueryContextAttributes(context_handle->ctxt_handle,
	                             SECPKG_ATTR_SIZES,
	                             &context_sizes);

	if (ret != SEC_E_OK || context_sizes.cbMaxSignature == 0)
		return GSS_S_FAILURE;

	input_desc.cBuffers = 2;
	input_desc.pBuffers = input_security_token;
	input_desc.ulVersion = SECBUFFER_VERSION;
	input_security_token[0].BufferType = SECBUFFER_DATA;
	input_security_token[0].cbBuffer = message_buffer->length;
	input_security_token[0].pvBuffer = message_buffer->value;
	input_security_token[1].BufferType = SECBUFFER_TOKEN;
	input_security_token[1].cbBuffer = context_sizes.cbMaxSignature;
	input_security_token[1].pvBuffer = xmalloc(context_sizes.cbMaxSignature * sizeof(char));

	ret = MakeSignature(context_handle->ctxt_handle,
	                    0,
	                    &input_desc,
	                    0);

	if (ret == SEC_E_OK) {
		msg_token->length = input_security_token[1].cbBuffer;
		msg_token->value = input_security_token[1].pvBuffer;
		return GSS_S_COMPLETE;
	}

	return GSS_S_FAILURE;

};

OM_uint32 gss_indicate_mechs (
        OM_uint32   *minor_status,
        gss_OID_set *mech_set)
{
	gss_OID_set set;

	if (minor_status != NULL)
		*minor_status = 0;

	if (mech_set == NULL)
		return GSS_S_CALL_INACCESSIBLE_WRITE;

	set = (gss_OID_set_desc *) xcalloc(1, sizeof(*set));

	/* We're only providing a single mech */
	set->count = 1;

	set->elements = (gss_OID_desc *) xcalloc(set->count, sizeof(*set->elements));

	set->elements[0].length = GSS_MECH_KRB5->length;
	set->elements[0].elements = xmalloc(GSS_MECH_KRB5->length);

	memcpy(set->elements[0].elements, GSS_MECH_KRB5->elements, GSS_MECH_KRB5->length);

	*mech_set = set;

	return GSS_S_COMPLETE;
};

#endif /* WIN32 */
