/***
  This file is part of PulseAudio.

  Copyright 2015 Pierre Ossman <ossman@cendio.se> for Cendio AB

  PulseAudio is free software; you can redistribute it and/or modify
  it under the terms of the GNU Lesser General Public License as published
  by the Free Software Foundation; either version 2.1 of the License,
  or (at your option) any later version.

  PulseAudio is distributed in the hope that it will be useful, but
  WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
  General Public License for more details.

  You should have received a copy of the GNU Lesser General Public License
  along with PulseAudio; if not, write to the Free Software
  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
  USA.
***/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <pulse/timeval.h>
#include <pulse/xmalloc.h>

#include <pulsecore/core-util.h>
#include <pulsecore/modargs.h>
#include <pulsecore/socket-client.h>
#include <pulsecore/strlist.h>

#include "module-tunnel-autoconnect-symdef.h"

PA_MODULE_AUTHOR("Pierre Ossman");
PA_MODULE_DESCRIPTION("Automatically reload module-tunnel-* when appropriate");
PA_MODULE_VERSION(PACKAGE_VERSION);
PA_MODULE_USAGE(
        "server=<address> "
        "cookie=<filename> "
        "app_name=<application name to send> "
        "app_id=<application id to send> "
        "app_version=<application version to send> "
        "app_icon=<application icon identifier to send>");

static const char* const valid_modargs[] = {
    "server",
    "cookie",
    "app_name",
    "app_id",
    "app_version",
    "app_icon",
    NULL,
};

struct userdata {
    pa_core *core;

    char *server;
    char *cookie;

    char *app_name;
    char *app_id;
    char *app_version;
    char *app_icon;

    pa_subscription *subscription;
    pa_socket_client *client;
    pa_time_event *ev;
    uint32_t sink, source;
};

static void time_event_cb(pa_mainloop_api *mainloop, pa_time_event* e, const struct timeval *tv, void *userdata);

static void on_connection(pa_socket_client *sc, pa_iochannel *io, void *userdata) {
    struct userdata *u = userdata;

    char *t;
    pa_module *m;

    pa_assert(sc);
    pa_assert(u);
    pa_assert(u->client == sc);

    pa_socket_client_unref(u->client);
    u->client = NULL;

    if (!io) {
        struct timeval tv;
        pa_gettimeofday(&tv);
        pa_timeval_add(&tv, 2000000);
        u->ev = u->core->mainloop->time_new(u->core->mainloop, &tv, time_event_cb, userdata);
        return;
    }

    pa_iochannel_free(io);

    t = pa_sprintf_malloc("server=\"%s\" cookie=\"%s\" app_name=\"%s\" app_id=\"%s\" app_version=\"%s\" app_icon=\"%s\"",
                          u->server, u->cookie, u->app_name, u->app_id,
                          u->app_version, u->app_icon);
    m = pa_module_load(u->core, "module-tunnel-sink-new", t);
    u->sink = m ? m->index : PA_INVALID_INDEX;
    m = pa_module_load(u->core, "module-tunnel-source-new", t);
    u->source = m ? m->index : PA_INVALID_INDEX;
    pa_xfree(t);
}

static void start_poll(struct userdata *u) {
    pa_strlist *server_list = NULL;
    char *server_name = NULL;

    if (!(server_list = pa_strlist_parse(u->server))) {
        pa_log("Invalid server specified.");
        return;
    }

    for (;;) {
        server_list = pa_strlist_pop(server_list, &server_name);

        if (server_name == NULL) {
            pa_log("Failed to connect to server '%s'", u->server);
            return;
        }

        if (!(u->client = pa_socket_client_new_string(u->core->mainloop, true, server_name, PA_NATIVE_DEFAULT_PORT)))
            continue;

        pa_xfree(server_name);
        pa_strlist_free(server_list);

        break;
    }

    pa_socket_client_set_callback(u->client, on_connection, u);
}

static void time_event_cb(pa_mainloop_api *mainloop, pa_time_event* e, const struct timeval *tv, void *userdata) {
    struct userdata *u = userdata;

    pa_assert(e);
    pa_assert(u);
    pa_assert(u->ev == e);

    mainloop->time_free(u->ev);
    u->ev = NULL;

    start_poll(u);
}

static void subscription_cb(pa_core *core, pa_subscription_event_type_t t, uint32_t idx, void *userdata) {
    struct userdata *u = userdata;

    pa_assert(u);

    if ((t & PA_SUBSCRIPTION_EVENT_TYPE_MASK) != PA_SUBSCRIPTION_EVENT_REMOVE)
        return;

    if (core->state == PA_CORE_SHUTDOWN)
        return;

    if (idx == u->sink)
        u->sink = PA_INVALID_INDEX;
    else if (idx == u->source)
        u->source = PA_INVALID_INDEX;
    else
        return;

    if ((u->sink == PA_INVALID_INDEX) && (u->source == PA_INVALID_INDEX))
        start_poll(u);
}

int pa__init(pa_module*m) {
    pa_modargs *ma = NULL;
    struct userdata *u;

    pa_assert(m);

    if (!(ma = pa_modargs_new(m->argument, valid_modargs))) {
        pa_log("Failed to parse module arguments");
        return -1;
    }

    m->userdata = u = pa_xnew0(struct userdata, 1);
    u->core = m->core;
    u->server = pa_xstrdup(pa_modargs_get_value(ma, "server", NULL));
    u->cookie = pa_xstrdup(pa_modargs_get_value(ma, "cookie", NULL));
    u->app_name = pa_xstrdup(pa_modargs_get_value(ma, "app_name", ""));;
    u->app_id = pa_xstrdup(pa_modargs_get_value(ma, "app_id", ""));;
    u->app_version = pa_xstrdup(pa_modargs_get_value(ma, "app_version", ""));;
    u->app_icon = pa_xstrdup(pa_modargs_get_value(ma, "app_icon", ""));;
    u->sink = PA_INVALID_INDEX;
    u->source = PA_INVALID_INDEX;

    pa_modargs_free(ma);

    if (u->server == NULL) {
        pa_log("No server specified.");
        pa__done(m);
        return -1;
    }

    u->subscription = pa_subscription_new(m->core, PA_SUBSCRIPTION_MASK_MODULE, subscription_cb, u);

    start_poll(u);

    return 0;
}

void pa__done(pa_module*m) {
    struct userdata *u;

    pa_assert(m);

    if (!(u = m->userdata))
        return;

    pa_xfree(u->server);
    pa_xfree(u->cookie);

    if (u->subscription != NULL)
        pa_subscription_free(u->subscription);
    if (u->client != NULL)
        pa_socket_client_unref(u->client);
    if (u->ev != NULL)
        m->core->mainloop->time_free(u->ev);

    pa_xfree(u);
}
