/* Copyright 2011-2021 Pierre Ossman <ossman@cendio.se> for Cendio AB
 * 
 * This is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this software; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307,
 * USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <assert.h>
#include <stdlib.h>
#include <list>

#include <rfb/encodings.h>

#if defined(HAVE_GNUTLS) || defined(HAVE_NETTLE)
#include <rfb/Security.h>
#include <rfb/SecurityClient.h>
#ifdef HAVE_GNUTLS
#include <rfb/CSecurityTLS.h>
#endif
#endif

#include "OptionsDialog.h"
#include "i18n.h"
#include "menukey.h"
#include "parameters.h"

#include "fltk/layout.h"
#include "fltk/util.h"
#include "fltk/Fl_Monitor_Arrangement.h"

#include <FL/Fl.H>
#include <FL/Fl_Button.H>
#include <FL/Fl_Check_Button.H>
#include <FL/Fl_Return_Button.H>
#include <FL/Fl_Round_Button.H>
#include <FL/Fl_Int_Input.H>
#include <FL/Fl_Choice.H>
#include "fltk/Fl_Navigation.h"

using namespace std;
using namespace rfb;

/* Match ThinLinc client options window size */
#define WINDOW_WIDTH    630
#define WINDOW_HEIGHT   460

#define DETAIL_BUTTON_WIDTH 84

std::map<OptionsCallback*, void*> OptionsDialog::callbacks;

static std::set<OptionsDialog *> instances;

OptionsDialog::OptionsDialog()
  : Fl_Window(WINDOW_WIDTH, WINDOW_HEIGHT, _("ThinLinc client options"))
{
  int x, y;
  Fl_Navigation *navigation;
  Fl_Button *button;

  // Odd dimensions to get rid of extra borders
  // FIXME: We need to retain the top border on Windows as they don't
  //        have any separator for the caption, which looks odd
#ifdef WIN32
  navigation = new Fl_Navigation(-1, 0, w()+2, h() - OUTER_MARGIN - BUTTON_HEIGHT - OUTER_MARGIN);
#else
  navigation = new Fl_Navigation(-1, -1, w()+2, h()+1 - OUTER_MARGIN - BUTTON_HEIGHT - OUTER_MARGIN);
#endif

  {
    int tx, ty, tw, th;

    navigation->client_area(tx, ty, tw, th, 150);

    createDisplayPage(tx, ty, tw, th);
    createLocalDevicesPage(tx, ty, tw, th);
    createOptimizationPage(tx, ty, tw, th);
    createSecurityPage(tx, ty, tw, th);
    createAdvancedPage(tx, ty, tw, th);
  }

  navigation->end();

  x = w() - BUTTON_WIDTH * 2 - INNER_MARGIN - OUTER_MARGIN;
  y = h() - BUTTON_HEIGHT - OUTER_MARGIN;

  button = new Fl_Button(x, y, BUTTON_WIDTH, BUTTON_HEIGHT, _("Cancel"));
  button->callback(this->handleCancel, this);

  x += BUTTON_WIDTH + INNER_MARGIN;

  button = new Fl_Return_Button(x, y, BUTTON_WIDTH, BUTTON_HEIGHT, _("OK"));
  button->callback(this->handleOK, this);

  callback(this->handleCancel, this);

  set_modal();

  if (instances.size() == 0)
    Fl::add_handler(fltk_event_handler);
  instances.insert(this);
}


OptionsDialog::~OptionsDialog()
{
  instances.erase(this);

  if (instances.size() == 0)
    Fl::remove_handler(fltk_event_handler);
}


void OptionsDialog::showDialog(void)
{
  static OptionsDialog *dialog = nullptr;

  if (!dialog)
    dialog = new OptionsDialog();

  if (dialog->shown())
    return;

  dialog->show();
}


void OptionsDialog::addCallback(OptionsCallback *cb, void *data)
{
  callbacks[cb] = data;
}


void OptionsDialog::removeCallback(OptionsCallback *cb)
{
  callbacks.erase(cb);
}


void OptionsDialog::show(void)
{
  /* show() gets called for raise events as well */
  if (!shown())
    loadOptions();

  Fl_Window::show();
}


void OptionsDialog::loadOptions(void)
{
  /* Display */
  if (!fullScreen) {
    windowed_button->setonly();
  } else {
    if (!strcasecmp(fullScreenMode, "all")) {
      full_screen_all_button->setonly();
    } else if (!strcasecmp(fullScreenMode, "selected")) {
      selected_monitors_button->setonly();
    } else {
      full_screen_current_button->setonly();
    }
  }

  monitor_arrangement->value(fullScreenSelectedMonitors.getParam());

  handleFullScreenMode(selected_monitors_button, this);

  /* Local devices */
  clipboard_checkbox->value(acceptClipboard && sendClipboard);

  if (_tl_featureLocalSound)
     sound_checkbox->value(_tl_soundRedirection);
  if (_tl_featureLocalDrives)
      drives_checkbox->value(_tl_driveRedirection);
  if (_tl_featureLocalPrinter)
      printers_checkbox->value(_tl_printerRedirection);
  if (_tl_featureLocalSmartcard)
      smartcard_checkbox->value(_tl_smartcardRedirection);

  /* Optimization */
  autoselectCheckbox->value(autoSelect);

  int encNum = encodingNum(preferredEncoding);

  switch (encNum) {
  case encodingTight:
    tightButton->setonly();
    break;
  case encodingZRLE:
    zrleButton->setonly();
    break;
  case encodingHextile:
    hextileButton->setonly();
    break;
  case encodingRaw:
    rawButton->setonly();
    break;
  }

  if (fullColour)
    fullcolorCheckbox->setonly();
  else {
    switch (lowColourLevel) {
    case 0:
      verylowcolorCheckbox->setonly();
      break;
    case 1:
      lowcolorCheckbox->setonly();
      break;
    case 2:
      mediumcolorCheckbox->setonly();
      break;
    }
  }

  char digit[2] = "0";

  compressionCheckbox->value(customCompressLevel);
  sshCompressionCheckbox->value(_tl_sshCompressionEnabled);
  jpegCheckbox->value(!noJpeg);
  digit[0] = '0' + compressLevel;
  compressionInput->value(digit);
  digit[0] = '0' + qualityLevel;
  jpegInput->value(digit);

  handleAutoselect(autoselectCheckbox, this);
  handleCompression(compressionCheckbox, this);
  handleJpeg(jpegCheckbox, this);

  /* Security */
  char buf[32];

  ssh_button[_tl_sshPortSelection]->setonly();
  snprintf(buf, sizeof(buf), "%d", (int)_tl_sshPortArbitrary);
  ssh_input->value(buf);

  if (!strcasecmp(_tl_authenticationMethod, "password"))
    password_button->setonly();
  else if (!strcasecmp(_tl_authenticationMethod, "publickey"))
    pubkey_button->setonly();
  else if (!strcasecmp(_tl_authenticationMethod, "scpublickey") &&
           _tl_featureSmartcardAuth)
    scpubkey_button->setonly();
  else if (!strcasecmp(_tl_authenticationMethod, "kerberos"))
    kerberos_button->setonly();

  /* Advanced */
  const char *menuKeyBuf;

  start_program_checkbox->value(_tl_startProgramEnabled);
  start_program_input->value(_tl_startProgramCommand);

  shadowing_checkbox->value(_tl_shadowingEnabled);

  emulate_mb_checkbox->value(emulateMiddleButton);
  syskeys_checkbox->value(fullscreenSystemKeys);

  menukey_choice->value(0);

  menuKeyBuf = menuKey;
  for (int idx = 0; idx < getMenuKeySymbolCount(); idx++)
    if (!strcmp(getMenuKeySymbols()[idx].name, menuKeyBuf))
      menukey_choice->value(idx + 1);

  if (!strcasecmp(_tl_reconnectPolicy, "single-disconnected"))
      auto_reconnect_button->setonly();
  else if (!strcasecmp(_tl_reconnectPolicy, "ask"))
      ask_reconnect_button->setonly();

  update_checkbox->value(_tl_updateEnabled);
}


void OptionsDialog::storeOptions(void)
{
  /* Display */
  if (windowed_button->value()) {
    fullScreen.setParam(false);
  } else {
    fullScreen.setParam(true);

    if (full_screen_all_button->value()) {
      fullScreenMode.setParam("All");
    } else if (selected_monitors_button->value()) {
      fullScreenMode.setParam("Selected");
    } else {
      fullScreenMode.setParam("Current");
    }
  }

  fullScreenSelectedMonitors.setParam(monitor_arrangement->value());

  /* Local device settings, except for clipboard sync, are disabled */
  acceptClipboard.setParam(clipboard_checkbox->value());
  sendClipboard.setParam(clipboard_checkbox->value());

  /* Optimization */
  autoSelect.setParam(autoselectCheckbox->value());

  if (tightButton->value())
    preferredEncoding.setParam(encodingName(encodingTight));
  else if (zrleButton->value())
    preferredEncoding.setParam(encodingName(encodingZRLE));
  else if (hextileButton->value())
    preferredEncoding.setParam(encodingName(encodingHextile));
  else if (rawButton->value())
    preferredEncoding.setParam(encodingName(encodingRaw));

  fullColour.setParam(fullcolorCheckbox->value());
  if (verylowcolorCheckbox->value())
    lowColourLevel.setParam(0);
  else if (lowcolorCheckbox->value())
    lowColourLevel.setParam(1);
  else if (mediumcolorCheckbox->value())
    lowColourLevel.setParam(2);

  customCompressLevel.setParam(compressionCheckbox->value());
  noJpeg.setParam(!jpegCheckbox->value());
  compressLevel.setParam(atoi(compressionInput->value()));
  qualityLevel.setParam(atoi(jpegInput->value()));

  /* Security settings are all disabled */

  /* Advanced */
  emulateMiddleButton.setParam(emulate_mb_checkbox->value());
  fullscreenSystemKeys.setParam(syskeys_checkbox->value());

  if (menukey_choice->value() == 0)
    menuKey.setParam("");
  else {
    menuKey.setParam(menukey_choice->text());
  }

  std::map<OptionsCallback*, void*>::const_iterator iter;

  for (iter = callbacks.begin();iter != callbacks.end();++iter)
    iter->first(iter->second);
}


void OptionsDialog::createLocalDevicesPage(int x, int y, int w, int h)
{
  Fl_Group *group = new Fl_Group(x, y, w, h, _("Local devices"));

  // Note: The layout of this page should be equal to the layout in
  // GraphicSessionPage_LocalDevices() in tlclient_optionwindow.cc

  int tx, ty;
  int width;

  tx = x + OUTER_MARGIN;
  ty = y + OUTER_MARGIN;

  /* Export local resources */
  ty += GROUP_LABEL_OFFSET;
  width = w - OUTER_MARGIN * 2;
  Fl_Group * res_box = new Fl_Group(tx, ty, width, 0,
                                    _("Export local resources"));
  res_box->labelfont(FL_BOLD);
  res_box->box(FL_FLAT_BOX);
  res_box->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    int details_x;

    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    width = res_box->w() - GROUP_LABEL_OFFSET * 2 - INNER_MARGIN - DETAIL_BUTTON_WIDTH;

    details_x = tx + width + INNER_MARGIN;

    // Checkbox for clipboard synchronization
    clipboard_checkbox = new Fl_Check_Button(LBLRIGHT(tx, ty, CHECK_MIN_WIDTH, CHECK_HEIGHT,
                                                      _("Clipboard synchronization")));
    clipboard_checkbox->tooltip(_("Synchronize clipboard between client and server"));

    ty += CHECK_HEIGHT + TIGHT_MARGIN;

    // FEATURE_LOCAL_SOUND
    if (_tl_featureLocalSound) {
      sound_checkbox = new Fl_Check_Button(LBLRIGHT(tx, ty, CHECK_MIN_WIDTH,
                                           CHECK_HEIGHT, _("Sound")));
      sound_checkbox->tooltip(_("Check if you want sound to be sent\n"
                                "from server to your terminal"));
      sound_checkbox->deactivate();
      // FEATURE_LOCAL_SOUND_ADVANCED
      if (_tl_featureLocalSoundAdvanced) {
        // Details dialog
        sound_advanced_button = new Fl_Button(details_x, ty,
                                              DETAIL_BUTTON_WIDTH, CHECK_HEIGHT,
                                              _("Details..."));
        sound_advanced_button->deactivate();
      }
      ty += CHECK_HEIGHT + TIGHT_MARGIN;
    }

    // FEATURE_LOCAL_DRIVES
    if (_tl_featureLocalDrives)  {
      // Checkbox for local drives.
      drives_checkbox= new Fl_Check_Button(LBLRIGHT(tx, ty, CHECK_MIN_WIDTH,
                                           CHECK_HEIGHT, _("Drives")));
      drives_checkbox->tooltip(_("Check to export local drives to server"));
      drives_checkbox->deactivate();

      // Details dialog for local drives.
      drives_advanced_button = new Fl_Button(details_x, ty,
                                             DETAIL_BUTTON_WIDTH, CHECK_HEIGHT,
                                             _("Details..."));
      drives_advanced_button->deactivate();

      ty += CHECK_HEIGHT + TIGHT_MARGIN;
    }

    // FEATURE_LOCAL_PRINTER
    if (_tl_featureLocalPrinter)  {
      // Checkbox for local printer.
      printers_checkbox= new Fl_Check_Button(LBLRIGHT(tx, ty, CHECK_MIN_WIDTH,
                                             CHECK_HEIGHT, _("Printer")));
      printers_checkbox->tooltip(_("Check to export local printer to server"));
      printers_checkbox->deactivate();

      // ADVANCED_LOCAL_PRINTER
      if (_tl_advancedLocalPrinter) {
        // Details dialog for local printers.
        printers_advanced_button = new Fl_Button(details_x, ty,
                                                 DETAIL_BUTTON_WIDTH,
                                                 CHECK_HEIGHT,
                                                 _("Details..."));
        printers_advanced_button->deactivate();
      }

      ty += CHECK_HEIGHT + TIGHT_MARGIN;
    }

    // FEATURE_LOCAL_SMARTCARD
    if (_tl_featureLocalSmartcard)  {
      smartcard_checkbox = new Fl_Check_Button(LBLRIGHT(tx, ty, CHECK_MIN_WIDTH,
                                               CHECK_HEIGHT,
                                               _("Smart card readers")));
      smartcard_checkbox->tooltip(
              _("Check to export local smart card readers to server"));
      smartcard_checkbox->deactivate();

      ty += CHECK_HEIGHT + TIGHT_MARGIN;
    }
  }
  ty += GROUP_LABEL_OFFSET - TIGHT_MARGIN;

  res_box->end();
  res_box->resizable(nullptr); // needed for resize to work sanely
  res_box->size(res_box->w(), ty - res_box->y());

  group->end();
}


void OptionsDialog::createOptimizationPage(int x, int y, int w, int h)
{
  Fl_Group *group = new Fl_Group(x, y, w, h, _("Optimization"));

  // Note: The layout of this page should be equal to the layout in
  // GraphicSessionPage_Optimization() in tlclient_optionwindow.cc

  int tx, ty;
  int orig_tx, orig_ty;
  int half_width, full_width;

  tx = x + OUTER_MARGIN;
  ty = y + OUTER_MARGIN;

  full_width = w - OUTER_MARGIN * 2;
  half_width = (full_width - INNER_MARGIN) / 2;

  /* AutoSelect checkbox */
  autoselectCheckbox = new Fl_Check_Button(LBLRIGHT(tx, ty,
                                                     CHECK_MIN_WIDTH,
                                                     CHECK_HEIGHT,
                                                     _("Auto select")));
  autoselectCheckbox->callback(handleAutoselect, this);
  ty += CHECK_HEIGHT + INNER_MARGIN;

  /* Two columns */
  orig_tx = tx;
  orig_ty = ty;

  /* VNC encoding box */
  ty += GROUP_LABEL_OFFSET;
  encodingGroup = new Fl_Group(tx, ty, half_width, 0,
                                _("Preferred encoding"));
  encodingGroup->box(FL_FLAT_BOX);
  encodingGroup->labelfont(FL_BOLD);
  encodingGroup->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    tightButton = new Fl_Round_Button(LBLRIGHT(tx, ty,
                                               RADIO_MIN_WIDTH,
                                               RADIO_HEIGHT,
                                               "Tight"));
    tightButton->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    zrleButton = new Fl_Round_Button(LBLRIGHT(tx, ty,
                                              RADIO_MIN_WIDTH,
                                              RADIO_HEIGHT,
                                              "ZRLE"));
    zrleButton->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    hextileButton = new Fl_Round_Button(LBLRIGHT(tx, ty,
                                                 RADIO_MIN_WIDTH,
                                                 RADIO_HEIGHT,
                                                 "Hextile"));
    hextileButton->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    rawButton = new Fl_Round_Button(LBLRIGHT(tx, ty,
                                             RADIO_MIN_WIDTH,
                                             RADIO_HEIGHT,
                                             "Raw"));
    rawButton->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;
  }

  ty += GROUP_LABEL_OFFSET - INNER_MARGIN;

  encodingGroup->end();
  encodingGroup->resizable(nullptr); // needed for resize to work sanely
  encodingGroup->size(encodingGroup->w(), ty - encodingGroup->y());

  /* Second column */
  tx = orig_tx + half_width + INNER_MARGIN;
  ty = orig_ty;

  /* Color box */
  ty += GROUP_LABEL_OFFSET;
  colorlevelGroup = new Fl_Group(tx, ty, half_width, 0, _("Color level"));
  colorlevelGroup->box(FL_FLAT_BOX);
  colorlevelGroup->labelfont(FL_BOLD);
  colorlevelGroup->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    fullcolorCheckbox = new Fl_Round_Button(LBLRIGHT(tx, ty,
                                                     RADIO_MIN_WIDTH,
                                                     RADIO_HEIGHT,
                                                     _("Full")));
    fullcolorCheckbox->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    mediumcolorCheckbox = new Fl_Round_Button(LBLRIGHT(tx, ty,
                                                       RADIO_MIN_WIDTH,
                                                       RADIO_HEIGHT,
                                                       _("Medium")));
    mediumcolorCheckbox->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    lowcolorCheckbox = new Fl_Round_Button(LBLRIGHT(tx, ty,
                                                    RADIO_MIN_WIDTH,
                                                    RADIO_HEIGHT,
                                                    _("Low")));
    lowcolorCheckbox->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    verylowcolorCheckbox = new Fl_Round_Button(LBLRIGHT(tx, ty,
                                                        RADIO_MIN_WIDTH,
                                                        RADIO_HEIGHT,
                                                        _("Very low")));
    verylowcolorCheckbox->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;
  }

  ty += GROUP_LABEL_OFFSET - INNER_MARGIN;

  colorlevelGroup->end();
  colorlevelGroup->resizable(nullptr); // needed for resize to work sanely
  colorlevelGroup->size(colorlevelGroup->w(), ty - colorlevelGroup->y());

  /* Back to normal */
  tx = orig_tx;
  ty += INNER_MARGIN;

  /* Checkboxes */
  compressionCheckbox = new Fl_Check_Button(LBLRIGHT(tx, ty,
                                                     CHECK_MIN_WIDTH,
                                                     CHECK_HEIGHT,
                                                     _("Custom compression level:")));
  compressionCheckbox->labelfont(FL_BOLD);
  compressionCheckbox->callback(handleCompression, this);
  ty += CHECK_HEIGHT + TIGHT_MARGIN;

  compressionInput = new Fl_Int_Input(tx + INDENT, ty,
                                      INPUT_HEIGHT, INPUT_HEIGHT,
                                      _("level (0=fast, 9=best)"));
  compressionInput->align(FL_ALIGN_RIGHT);
  ty += INPUT_HEIGHT + INNER_MARGIN;

  jpegCheckbox = new Fl_Check_Button(LBLRIGHT(tx, ty,
                                              CHECK_MIN_WIDTH,
                                              CHECK_HEIGHT,
                                              _("Allow JPEG compression:")));
  jpegCheckbox->labelfont(FL_BOLD);
  jpegCheckbox->callback(handleJpeg, this);
  ty += CHECK_HEIGHT + TIGHT_MARGIN;

  jpegInput = new Fl_Int_Input(tx + INDENT, ty,
                               INPUT_HEIGHT, INPUT_HEIGHT,
                               _("quality (0=poor, 9=best)"));
  jpegInput->align(FL_ALIGN_RIGHT);
  ty += INPUT_HEIGHT + INNER_MARGIN;

  sshCompressionCheckbox = new Fl_Check_Button(LBLRIGHT(tx, ty,
          RADIO_MIN_WIDTH, CHECK_HEIGHT,
          _("SSH compression")));
  sshCompressionCheckbox->deactivate();
  ty += CHECK_HEIGHT + INNER_MARGIN;

  /* Back to normal */
  tx = orig_tx;
  ty += INNER_MARGIN;

  group->end();
}


void OptionsDialog::createSecurityPage(int x, int y, int w, int h)
{
  Fl_Group *group = new Fl_Group(x, y, w, h, _("Security"));

  // Note: The layout of this page should be equal to the layout in
  // GraphicSessionPage_Security() in tlclient_optionwindow.cc

  int tx, ty;
  int width;

  int idx;

  tx = x + OUTER_MARGIN;
  ty = y + OUTER_MARGIN;

  // SSH port
  ty += GROUP_LABEL_OFFSET;
  width = w - OUTER_MARGIN * 2;
  Fl_Group * port_box = new Fl_Group(tx, ty, width, 0, _("SSH port"));
  port_box->labelfont(FL_BOLD);
  port_box->box(FL_FLAT_BOX);
  port_box->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    width = port_box->w() - GROUP_LABEL_OFFSET * 2;

    ssh_button[0] = new Fl_Round_Button(LBLRIGHT(tx, ty, RADIO_MIN_WIDTH,
                                        RADIO_HEIGHT,
                                        _("22 (default SSH port)")));
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    ssh_button[1] = new Fl_Round_Button(LBLRIGHT(tx, ty, RADIO_MIN_WIDTH,
                                        RADIO_HEIGHT, _("80 (HTTP)")));
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    ssh_button[2] = new Fl_Round_Button(tx, ty, RADIO_MIN_WIDTH,
                                        RADIO_HEIGHT, "");

    ssh_input = new Fl_Int_Input(tx + RADIO_MIN_WIDTH + TIGHT_MARGIN, ty,
                                 140, CHECK_HEIGHT);
    ssh_input->tooltip(_("Arbitary SSH port number"));

    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    /* Fix up attributes */
    for (idx = 0; idx < 3; idx++) {
      ssh_button[idx]->type(FL_RADIO_BUTTON);
    }
  }

  ty += GROUP_LABEL_OFFSET - TIGHT_MARGIN;

  port_box->end();
  port_box->deactivate();
  port_box->resizable(nullptr); // needed for resize to work sanely
  port_box->size(port_box->w(), ty - port_box->y());

  tx = x + OUTER_MARGIN;
  ty += INNER_MARGIN;

  // Authentication method
  ty += GROUP_LABEL_OFFSET;
  width = w - OUTER_MARGIN * 2;
  Fl_Group * auth_method_group = new Fl_Group(tx, ty, width, 0,
                                              _("Authentication method"));
  auth_method_group->labelfont(FL_BOLD);
  auth_method_group->box(FL_FLAT_BOX);
  auth_method_group->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    width = port_box->w() - GROUP_LABEL_OFFSET * 2;

    password_button = new Fl_Round_Button(LBLRIGHT(tx, ty, RADIO_MIN_WIDTH,
                                          RADIO_HEIGHT, _("Password")));
    password_button->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    pubkey_button = new Fl_Round_Button(LBLRIGHT(tx, ty, RADIO_MIN_WIDTH,
                                                 RADIO_HEIGHT,
                                                 _("Public key")));
    pubkey_button->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    if (_tl_featureSmartcardAuth) {
      scpubkey_button = new Fl_Round_Button(LBLRIGHT(tx, ty, RADIO_MIN_WIDTH,
                                                     RADIO_HEIGHT,
                                                     _("Smart card")));
      scpubkey_button->type(FL_RADIO_BUTTON);

      int details_x = auth_method_group->x()  + auth_method_group->w() -
          GROUP_LABEL_OFFSET - DETAIL_BUTTON_WIDTH;

      smartcard_trigger_details_button = new Fl_Button(details_x, ty,
                                                       DETAIL_BUTTON_WIDTH,
                                                       CHECK_HEIGHT,
                                                       _("Details..."));

      ty += RADIO_HEIGHT + TIGHT_MARGIN;
    }

    kerberos_button = new Fl_Round_Button(LBLRIGHT(tx, ty, RADIO_MIN_WIDTH,
                                                   RADIO_HEIGHT,
                                                   _("Kerberos ticket")));
    kerberos_button->type(FL_RADIO_BUTTON);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;
  }
  ty += GROUP_LABEL_OFFSET - TIGHT_MARGIN;

  auth_method_group->end();
  auth_method_group->deactivate();
  auth_method_group->resizable(nullptr); // needed for resize to work sanely
  auth_method_group->size(auth_method_group->w(), ty -
                          auth_method_group->y());

  tx = x + OUTER_MARGIN;
  ty += INNER_MARGIN;

  group->end();
}


void OptionsDialog::createAdvancedPage(int x, int y, int w, int h)
{
  Fl_Group *group = new Fl_Group(x, y, w, h, _("Advanced"));

  // Note: The layout of this page should be equal to the layout in
  // GraphicSessionPage_Advanced() in tlclient_optionwindow.cc

  int tx, ty;
  int full_width;
  int width;

  tx = x + OUTER_MARGIN;
  ty = y + OUTER_MARGIN;

  full_width = w - OUTER_MARGIN * 2;

  // Start program
  ty += GROUP_LABEL_OFFSET;
  Fl_Group * start_program_box = new Fl_Group(tx, ty, full_width, 0,
                                              _("Start a program"));
  start_program_box->labelfont(FL_BOLD);
  start_program_box->box(FL_FLAT_BOX);
  start_program_box->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    width = start_program_box->w() - GROUP_LABEL_OFFSET * 2;

    start_program_checkbox = new Fl_Check_Button(
            LBLRIGHT(tx, ty, CHECK_MIN_WIDTH, CHECK_HEIGHT,
                     _("Start the following program")));
    start_program_checkbox->tooltip(
            _("Check if you want to start the session "
              "with the specified command"));
    ty += CHECK_HEIGHT + INNER_MARGIN;

    start_program_input = new Fl_Input(LBLLEFT(tx, ty,
                                       width, INPUT_HEIGHT,
                                       _("Command:")));
    ty += INPUT_HEIGHT + INNER_MARGIN;
  }

  ty += GROUP_LABEL_OFFSET - INNER_MARGIN;

  start_program_box->end();
  start_program_box->deactivate();
  start_program_box->resizable(nullptr); // needed for resize to work sanely
  start_program_box->size(start_program_box->w(), ty - start_program_box->y());

  tx = x + OUTER_MARGIN;
  ty += INNER_MARGIN;

  /* opt_box: client options. */
  ty += GROUP_LABEL_OFFSET;
  Fl_Group * opt_box = new Fl_Group(tx, ty, full_width, 0,
                                    _("Session options"));
  opt_box->labelfont(FL_BOLD);
  opt_box->box(FL_FLAT_BOX);
  opt_box->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    width = opt_box->w() - GROUP_LABEL_OFFSET * 2;

    shadowing_checkbox = new Fl_Check_Button(
            LBLRIGHT(tx, ty, CHECK_MIN_WIDTH, CHECK_HEIGHT,
                     _("Enable shadowing")));
    shadowing_checkbox->tooltip(_("Check if you want to enable user shadowing"));
    shadowing_checkbox->deactivate();
    ty += CHECK_HEIGHT + TIGHT_MARGIN;

    emulate_mb_checkbox = new Fl_Check_Button(
            LBLRIGHT(tx, ty, CHECK_MIN_WIDTH, CHECK_HEIGHT,
                     _("Emulate middle mouse button")));
    ty += CHECK_HEIGHT + TIGHT_MARGIN;

    syskeys_checkbox = new Fl_Check_Button(
            LBLRIGHT(tx, ty, CHECK_MIN_WIDTH, CHECK_HEIGHT,
                     _("Send system keys")));
    syskeys_checkbox->tooltip(_("Check if you want system keys (like "
                                "Alt+Tab) to be sent to the server when "
                                "in full-screen mode"));
    ty += CHECK_HEIGHT + INNER_MARGIN;

    menukey_choice = new Fl_Choice(LBLLEFT(tx, ty, width/2, CHOICE_HEIGHT,
                                           _("Popup menu key")));

    fltk_menu_add(menukey_choice, _("None"), 0, nullptr, nullptr,
                  FL_MENU_DIVIDER);
    for (int idx = 0; idx < getMenuKeySymbolCount(); idx++)
      fltk_menu_add(menukey_choice, getMenuKeySymbols()[idx].name,
                    0, nullptr, nullptr, 0);

    ty += CHOICE_HEIGHT + INNER_MARGIN;
  }

  ty += GROUP_LABEL_OFFSET - INNER_MARGIN;

  opt_box->end();
  opt_box->resizable(nullptr); // needed for resize to work sanely
  opt_box->size(opt_box->w(), ty - opt_box->y());

  /* Restore position */
  tx = x + OUTER_MARGIN;
  ty += INNER_MARGIN;

  /* opt_box: reconnect policy */
  ty += GROUP_LABEL_OFFSET;
  Fl_Group * reconnect_group = new Fl_Group(tx, ty, full_width, 0,
                                            _("Reconnect policy"));
  reconnect_group->labelfont(FL_BOLD);
  reconnect_group->box(FL_FLAT_BOX);
  reconnect_group->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    width = reconnect_group->w() - GROUP_LABEL_OFFSET * 2;

    auto_reconnect_button = new Fl_Round_Button(
            LBLRIGHT(tx, ty, RADIO_MIN_WIDTH, RADIO_HEIGHT,
            _("Automatically reconnect to a disconnected session")));
    auto_reconnect_button->type(FL_RADIO_BUTTON);
    auto_reconnect_button->tooltip(
        _("1. If there is no disconnected session and additional sessions are allowed, create a new session.\n"
          "2. If there is a single disconnected session, or if server allows only one session, reconnect to existing session.\n"
          "3. Otherwise, ask how to proceed."));
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    ask_reconnect_button = new Fl_Round_Button(
            LBLRIGHT(tx, ty, RADIO_MIN_WIDTH, RADIO_HEIGHT,
            _("Always ask how multiple sessions should be handled")));
    ask_reconnect_button->type(FL_RADIO_BUTTON);
    ask_reconnect_button->tooltip(
        _("1. If there is no running session, create a new session.\n"
          "2. If server allows only one session, reconnect to existing session.\n"
          "3. Otherwise, ask how to proceed."));
    ty += RADIO_HEIGHT + TIGHT_MARGIN;
  }

  ty += GROUP_LABEL_OFFSET - TIGHT_MARGIN;

  reconnect_group->end();
  reconnect_group->deactivate();
  reconnect_group->resizable(nullptr); // needed for resize to work sanely
  reconnect_group->size(reconnect_group->w(), ty - reconnect_group->y());

  tx = x + OUTER_MARGIN;
  ty += INNER_MARGIN;

  /* opt_box: update */
  ty += GROUP_LABEL_OFFSET;
  Fl_Group * update_group = new Fl_Group(tx, ty, full_width, 0,
                                         _("Software updates"));
  update_group->labelfont(FL_BOLD);
  update_group->box(FL_FLAT_BOX);
  update_group->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    width = update_group->w() - GROUP_LABEL_OFFSET * 2;

    update_checkbox = new Fl_Check_Button(
            LBLRIGHT(tx, ty, CHECK_MIN_WIDTH, CHECK_HEIGHT,
            _("Check for client updates")));
    update_checkbox->tooltip(
            _("If enabled, client will regularly check for new versions"));
    ty += CHECK_HEIGHT + INNER_MARGIN;
  }

  ty += GROUP_LABEL_OFFSET - INNER_MARGIN;

  update_group->end();
  update_group->deactivate();
  update_group->resizable(nullptr); // needed for resize to work sanely
  update_group->size(update_group->w(), ty - update_group->y());

  tx = x + OUTER_MARGIN;
  ty += INNER_MARGIN;

  group->end();
}


void OptionsDialog::createDisplayPage(int x, int y, int w, int h)
{
  Fl_Group *group = new Fl_Group(x, y, w, h, _("Display"));

  // Note: The layout of this page should be equal to the layout in
  // GraphicSessionPage_Display() in tlclient_optionwindow.cc

  int tx, ty, width;

  tx = x + OUTER_MARGIN;
  ty = y + OUTER_MARGIN;

  /* Display mode */
  ty += GROUP_LABEL_OFFSET;
  width = w - OUTER_MARGIN * 2;
  display_group = new Fl_Group(tx, ty, width, 0, _("Display mode"));
  display_group->box(FL_FLAT_BOX);
  display_group->labelfont(FL_BOLD);
  display_group->align(FL_ALIGN_LEFT | FL_ALIGN_TOP);

  {
    tx += GROUP_LABEL_OFFSET;
    ty += TIGHT_MARGIN;

    width = display_group->w() - GROUP_LABEL_OFFSET * 2;

    windowed_button = new Fl_Round_Button(LBLRIGHT(tx, ty,
            RADIO_MIN_WIDTH, RADIO_HEIGHT,
            _("Windowed")));
    windowed_button->type(FL_RADIO_BUTTON);
    windowed_button->callback(handleFullScreenMode, this);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    full_screen_current_button = new Fl_Round_Button(LBLRIGHT(tx, ty,
            RADIO_MIN_WIDTH, RADIO_HEIGHT,
            _("Full screen on current monitor")));
    full_screen_current_button->type(FL_RADIO_BUTTON);
    full_screen_current_button->callback(handleFullScreenMode, this);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    full_screen_all_button = new Fl_Round_Button(LBLRIGHT(tx, ty,
            RADIO_MIN_WIDTH, RADIO_HEIGHT,
            _("Full screen on all monitors")));
    full_screen_all_button->type(FL_RADIO_BUTTON);
    full_screen_all_button->callback(handleFullScreenMode, this);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    selected_monitors_button = new Fl_Round_Button(LBLRIGHT(tx, ty,
            RADIO_MIN_WIDTH, RADIO_HEIGHT,
            _("Full screen on selected monitor(s)")));
    selected_monitors_button->type(FL_RADIO_BUTTON);
    selected_monitors_button->callback(handleFullScreenMode, this);
    ty += RADIO_HEIGHT + TIGHT_MARGIN;

    monitor_arrangement = new Fl_Monitor_Arrangement(tx + INDENT, ty,
                                                     width - INDENT,
                                                     150);
    ty += 150 + TIGHT_MARGIN;
  }
  ty += GROUP_LABEL_OFFSET - TIGHT_MARGIN;

  display_group->end();
  display_group->resizable(nullptr); // needed for resize to work sanely
  display_group->size(display_group->w(), ty - display_group->y());

  group->end();
}


void OptionsDialog::handleAutoselect(Fl_Widget* /*widget*/, void *data)
{
  OptionsDialog *dialog = (OptionsDialog*)data;

  if (dialog->autoselectCheckbox->value()) {
    dialog->encodingGroup->deactivate();
    dialog->colorlevelGroup->deactivate();
  } else {
    dialog->encodingGroup->activate();
    dialog->colorlevelGroup->activate();
  }

  // JPEG setting is also affected by autoselection
  dialog->handleJpeg(dialog->jpegCheckbox, dialog);
}


void OptionsDialog::handleCompression(Fl_Widget* /*widget*/, void *data)
{
  OptionsDialog *dialog = (OptionsDialog*)data;

  if (dialog->compressionCheckbox->value())
    dialog->compressionInput->activate();
  else
    dialog->compressionInput->deactivate();
}


void OptionsDialog::handleJpeg(Fl_Widget* /*widget*/, void *data)
{
  OptionsDialog *dialog = (OptionsDialog*)data;

  if (dialog->jpegCheckbox->value() &&
      !dialog->autoselectCheckbox->value())
    dialog->jpegInput->activate();
  else
    dialog->jpegInput->deactivate();
}


void OptionsDialog::handleFullScreenMode(Fl_Widget* /*widget*/, void *data)
{
  OptionsDialog *dialog = (OptionsDialog*)data;

  if (dialog->selected_monitors_button->value()) {
    dialog->monitor_arrangement->activate();
  } else {
    dialog->monitor_arrangement->deactivate();
  }
}

void OptionsDialog::handleCancel(Fl_Widget* /*widget*/, void *data)
{
  OptionsDialog *dialog = (OptionsDialog*)data;

  dialog->hide();
}


void OptionsDialog::handleOK(Fl_Widget* /*widget*/, void *data)
{
  OptionsDialog *dialog = (OptionsDialog*)data;

  dialog->hide();

  dialog->storeOptions();
}

int OptionsDialog::fltk_event_handler(int event)
{
  std::set<OptionsDialog *>::iterator iter;

  if (event != FL_SCREEN_CONFIGURATION_CHANGED)
    return 0;

  // Refresh monitor arrangement widget to match the parameter settings after
  // screen configuration has changed. The MonitorArrangement index doesn't work
  // the same way as the FLTK screen index.
  for (iter = instances.begin(); iter != instances.end(); iter++)
      Fl::add_timeout(0, handleScreenConfigTimeout, (*iter));

  return 0;
}

void OptionsDialog::handleScreenConfigTimeout(void *data)
{
    OptionsDialog *self = (OptionsDialog *)data;

    assert(self);

    self->monitor_arrangement->value(fullScreenSelectedMonitors.getParam());
}
